<?php
declare(strict_types=1);

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/app.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();
csrf_check();

// moradores ativos para seleção
$moradores = $pdo->query("
  SELECT m.id, m.nome, q.codigo AS quarto
  FROM moradores m
  JOIN quartos q ON q.id = m.quarto_id
  WHERE m.ativo=1
  ORDER BY m.nome
")->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $morador_id = (int)($_POST['morador_id'] ?? 0);
  $nome_visitante = trim((string)($_POST['nome_visitante'] ?? ''));
  $documento = trim((string)($_POST['documento'] ?? ''));
  $telefone = trim((string)($_POST['telefone'] ?? ''));
  $observacao = trim((string)($_POST['observacao'] ?? ''));

  if ($morador_id <= 0 || $nome_visitante === '') {
    flash_set('error', 'Informe o morador e o nome do visitante.');
    redirect('create.php');
  }

  try {
    $pdo->prepare("
      INSERT INTO visitas (morador_id, nome_visitante, documento, telefone, observacao, data_entrada)
      VALUES (?,?,?,?,?, NOW())
    ")->execute([
      $morador_id,
      $nome_visitante,
      $documento ?: null,
      $telefone ?: null,
      $observacao ?: null
    ]);

    flash_set('success', 'Visita registrada.');
    redirect('index.php');

  } catch (Throwable $e) {
    flash_set('error', 'Erro: ' . $e->getMessage());
    redirect('create.php');
  }
}

require_once __DIR__ . '/../../includes/header.php';
require_once __DIR__ . '/../../includes/sidebar.php';
?>

<div class="card">
  <div style="font-weight:900;font-size:18px">Registrar visita</div>
  <div class="sep"></div>

  <form method="post" class="form-grid cols-2" id="formVisita">
    <input type="hidden" name="csrf" value="<?= htmlspecialchars(csrf_token()) ?>">

    <div style="grid-column:1/-1">
      <label>Pesquisar morador</label>
      <input class="input" id="busca" placeholder="Digite o nome do morador para filtrar…">
      <small style="color:var(--muted)">Digite para filtrar a lista abaixo.</small>
    </div>

    <div style="grid-column:1/-1">
      <label>Morador</label>
      <select class="input" name="morador_id" id="moradorSelect" required size="8" style="height:auto">
        <?php foreach ($moradores as $m): ?>
          <option value="<?= (int)$m['id'] ?>">
            <?= htmlspecialchars($m['nome']) ?> — Quarto <?= htmlspecialchars($m['quarto']) ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>

    <div style="grid-column:1/-1">
      <label>Nome do visitante</label>
      <input class="input" name="nome_visitante" required>
    </div>

    <div>
      <label>Documento (opcional)</label>
      <input class="input" name="documento">
    </div>

    <div>
      <label>Telefone (opcional)</label>
      <input class="input" name="telefone" placeholder="(DDD) 9XXXX-XXXX">
    </div>

    <div style="grid-column:1/-1">
      <label>Observação</label>
      <input class="input" name="observacao" placeholder="Ex: visita do fim de semana">
    </div>

    <div class="row" style="grid-column:1/-1">
      <button class="btn" type="submit">Salvar</button>
      <a class="btn btn-ghost" href="index.php">Cancelar</a>
    </div>
  </form>
</div>

<script>
(function(){
  const busca = document.getElementById('busca');
  const sel = document.getElementById('moradorSelect');
  if (!busca || !sel) return;

  const all = Array.from(sel.options).map(o => ({value:o.value, text:o.text}));

  function render(list){
    sel.innerHTML = '';
    list.forEach(item => {
      const opt = document.createElement('option');
      opt.value = item.value;
      opt.textContent = item.text;
      sel.appendChild(opt);
    });
  }

  busca.addEventListener('input', function(){
    const q = this.value.trim().toLowerCase();
    if (!q) return render(all);
    const filtered = all.filter(o => o.text.toLowerCase().includes(q));
    render(filtered);
  });
})();
</script>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
