<?php
declare(strict_types=1);

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/app.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

// ✅ view.php deve ser GET
$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) {
  http_response_code(404);
  exit('Morador inválido.');
}

// Morador + quarto
$st = $pdo->prepare("
  SELECT m.*,
         q.codigo AS quarto_codigo,
         q.banheiro_privativo
  FROM moradores m
  JOIN quartos q ON q.id = m.quarto_id
  WHERE m.id = ?
");
$st->execute([$id]);
$m = $st->fetch();

if (!$m) {
  http_response_code(404);
  exit('Morador não encontrado.');
}

// Pagamentos recentes
$pagamentos = [];
try {
  $stP = $pdo->prepare("
    SELECT id, tipo, data_pagamento, referencia_mes, referencia_ano, valor_base, multa, desconto, total, forma
    FROM pagamentos
    WHERE morador_id = ?
    ORDER BY data_pagamento DESC, id DESC
    LIMIT 20
  ");
  $stP->execute([$id]);
  $pagamentos = $stP->fetchAll();
} catch (Throwable $e) {
  // se a tabela não tiver todas as colunas, não quebra o view
  $pagamentos = [];
}

// Advertências recentes (se existir tabela)
$advertencias = [];
$adv_count = 0;
try {
  $stA = $pdo->prepare("
    SELECT id, tipo, data, descricao, ativa
    FROM advertencias
    WHERE morador_id = ?
    ORDER BY data DESC, id DESC
    LIMIT 20
  ");
  $stA->execute([$id]);
  $advertencias = $stA->fetchAll();

  $stAC = $pdo->prepare("SELECT COUNT(*) FROM advertencias WHERE morador_id=? AND ativa=1");
  $stAC->execute([$id]);
  $adv_count = (int)$stAC->fetchColumn();
} catch (Throwable $e) {
  $advertencias = [];
  $adv_count = 0;
}

// WhatsApp links
function only_digits(?string $v): string {
  return preg_replace('/\D+/', '', (string)$v);
}
$tel_contato = only_digits($m['telefone_contato'] ?? '');
$tel_emerg   = only_digits($m['telefone_emergencia'] ?? '');
$wpp_contato = $tel_contato ? ("https://wa.me/55" . $tel_contato) : null;
$wpp_emerg   = $tel_emerg ? ("https://wa.me/55" . $tel_emerg) : null;

require_once __DIR__ . '/../../includes/header.php';
require_once __DIR__ . '/../../includes/sidebar.php';
?>

<div class="row" style="justify-content:space-between;align-items:center">
  <div>
    <div style="font-weight:900;font-size:18px"><?= htmlspecialchars($m['nome']) ?></div>
    <div style="color:var(--muted);font-size:13px;margin-top:4px">
      Quarto <b><?= htmlspecialchars((string)$m['quarto_codigo']) ?></b>
      <?php if ((int)$m['banheiro_privativo'] === 1): ?>
        • <span class="badge ok">Banheiro privativo</span>
      <?php else: ?>
        • <span class="badge">Banheiro compartilhado</span>
      <?php endif; ?>

      <?php if ((int)$m['ativo'] === 1): ?>
        • <span class="badge ok">Ativo</span>
      <?php else: ?>
        • <span class="badge danger">Encerrado</span>
      <?php endif; ?>

      <?php if ($adv_count > 0): ?>
        • <span class="badge warn"><?= $adv_count ?> advertência(s) ativa(s)</span>
      <?php endif; ?>
    </div>
  </div>

  <div class="row">
    <a class="btn btn-ghost" href="/pensionato/modules/moradores/index.php">Voltar</a>
    <a class="btn" href="/pensionato/modules/pagamentos/create.php?morador_id=<?= (int)$m['id'] ?>">+ Pagamento</a>
    <a class="btn btn-ghost" href="/pensionato/modules/moradores/edit.php?id=<?= (int)$m['id'] ?>">Editar</a>
  </div>
</div>

<div class="sep"></div>

<div class="grid cols-2">
  <!-- Dados principais -->
  <div class="card">
    <div style="font-weight:900">Dados do morador</div>
    <div class="sep"></div>

    <div class="form-grid cols-2">
      <div>
        <label>Nome</label>
        <div class="input" style="display:flex;align-items:center"><?= htmlspecialchars((string)$m['nome']) ?></div>
      </div>

      <div>
        <label>Quarto</label>
        <div class="input" style="display:flex;align-items:center"><b><?= htmlspecialchars((string)$m['quarto_codigo']) ?></b></div>
      </div>

      <div>
        <label>Entrada</label>
        <div class="input" style="display:flex;align-items:center"><?= htmlspecialchars((string)($m['data_entrada'] ?? '—')) ?></div>
      </div>

      <div>
        <label>Saída</label>
        <div class="input" style="display:flex;align-items:center"><?= htmlspecialchars((string)($m['data_saida'] ?? '—')) ?></div>
      </div>

      <div>
        <label>Valor mensal</label>
        <div class="input" style="display:flex;align-items:center"><b><?= money((float)($m['valor_mensal'] ?? 0)) ?></b></div>
      </div>

      <div>
        <label>Pessoas no quarto</label>
        <div class="input" style="display:flex;align-items:center"><?= (int)($m['qtd_pessoas'] ?? 1) ?></div>
      </div>

      <div style="grid-column:1/-1">
        <label>Observações</label>
        <div class="input" style="display:flex;align-items:center"><?= htmlspecialchars((string)($m['observacao'] ?? '—')) ?></div>
      </div>
    </div>
  </div>

  <!-- Contatos / Facial -->
  <div class="card">
    <div style="font-weight:900">Contato e Facial</div>
    <div class="sep"></div>

    <div class="form-grid cols-2">
      <div>
        <label>Telefone de contato</label>
        <div class="row">
          <div class="input" style="flex:1;display:flex;align-items:center"><?= htmlspecialchars((string)($m['telefone_contato'] ?? '—')) ?></div>
          <?php if ($wpp_contato): ?>
            <a class="btn" target="_blank" href="<?= htmlspecialchars($wpp_contato) ?>">WhatsApp</a>
          <?php endif; ?>
        </div>
      </div>

      <div>
        <label>Falar com</label>
        <div class="input" style="display:flex;align-items:center"><?= htmlspecialchars((string)($m['falar_com'] ?? '—')) ?></div>
      </div>

      <div>
        <label>Telefone de emergência</label>
        <div class="row">
          <div class="input" style="flex:1;display:flex;align-items:center"><?= htmlspecialchars((string)($m['telefone_emergencia'] ?? '—')) ?></div>
          <?php if ($wpp_emerg): ?>
            <a class="btn" target="_blank" href="<?= htmlspecialchars($wpp_emerg) ?>">WhatsApp</a>
          <?php endif; ?>
        </div>
      </div>

      <div>
        <label>Facial cadastrada?</label>
        <div class="input" style="display:flex;align-items:center">
          <?php if ((int)($m['facial_cadastrada'] ?? 0) === 1): ?>
            <span class="badge ok">Sim</span>
          <?php else: ?>
            <span class="badge warn">Não</span>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
</div>

<div class="sep"></div>

<!-- Pagamentos -->
<div class="card">
  <div class="row" style="justify-content:space-between;align-items:center">
    <div style="font-weight:900">Pagamentos recentes</div>
    <a class="btn btn-ghost" href="/pensionato/modules/pagamentos/index.php?morador_id=<?= (int)$m['id'] ?>">Ver todos</a>
  </div>

  <div class="sep"></div>

  <?php if (!$pagamentos): ?>
    <div style="color:var(--muted)">Nenhum pagamento encontrado.</div>
  <?php else: ?>
    <table class="table">
      <thead>
        <tr>
          <th>Data</th>
          <th>Tipo</th>
          <th>Ref</th>
          <th>Forma</th>
          <th>Total</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($pagamentos as $p): ?>
          <tr>
            <td><?= htmlspecialchars((string)$p['data_pagamento']) ?></td>
            <td><?= htmlspecialchars((string)$p['tipo']) ?></td>
            <td>
              <?php if (($p['tipo'] ?? '') === 'mensalidade'): ?>
                <?= str_pad((string)$p['referencia_mes'], 2, '0', STR_PAD_LEFT) ?>/<?= (int)$p['referencia_ano'] ?>
              <?php else: ?>
                —
              <?php endif; ?>
            </td>
            <td><?= htmlspecialchars((string)($p['forma'] ?? '—')) ?></td>
            <td><b><?= money((float)($p['total'] ?? 0)) ?></b></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  <?php endif; ?>
</div>

<div class="sep"></div>

<!-- Advertências -->
<div class="card">
  <div class="row" style="justify-content:space-between;align-items:center">
    <div style="font-weight:900">Advertências</div>
    <a class="btn btn-ghost" href="/pensionato/modules/advertencias/create.php?morador_id=<?= (int)$m['id'] ?>">+ Advertência</a>
  </div>

  <div class="sep"></div>

  <?php if (!$advertencias): ?>
    <div style="color:var(--muted)">Nenhuma advertência registrada.</div>
  <?php else: ?>
    <table class="table">
      <thead>
        <tr>
          <th>Data</th>
          <th>Tipo</th>
          <th>Status</th>
          <th>Descrição</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($advertencias as $a): ?>
          <tr>
            <td><?= htmlspecialchars((string)$a['data']) ?></td>
            <td><?= htmlspecialchars((string)$a['tipo']) ?></td>
            <td>
              <?php if ((int)($a['ativa'] ?? 0) === 1): ?>
                <span class="badge warn">Ativa</span>
              <?php else: ?>
                <span class="badge">Resolvida</span>
              <?php endif; ?>
            </td>
            <td><?= htmlspecialchars((string)($a['descricao'] ?? '—')) ?></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  <?php endif; ?>
</div>

<div class="sep"></div>

<!-- Encerrar estadia (POST) -->
<?php if ((int)$m['ativo'] === 1): ?>
  <div class="card">
    <div style="font-weight:900;color:#ffb4b4">Encerrar estadia</div>
    <div class="sep"></div>

    <form method="post"
          action="/pensionato/modules/moradores/delete.php"
          onsubmit="return confirm('Encerrar a estadia deste morador?');">
      <input type="hidden" name="csrf" value="<?= htmlspecialchars(csrf_token()) ?>">
      <input type="hidden" name="id" value="<?= (int)$m['id'] ?>">

      <div class="form-grid cols-2">
        <div>
          <label>Data de saída</label>
          <input class="input" type="date" name="data_saida" value="<?= date('Y-m-d') ?>">
        </div>
        <div style="display:flex;align-items:flex-end">
          <button class="btn danger" type="submit">Encerrar</button>
        </div>
      </div>
    </form>
  </div>
<?php endif; ?>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
