<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/app.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();
csrf_check();

function calc_valor(int $banheiro_privativo, int $qtd): float {
  $qtd = ($qtd === 2) ? 2 : 1;
  if ($banheiro_privativo === 1) return PRECO['privativo'][$qtd];
  return PRECO['compartilhado'][$qtd];
}

$id = (int)($_GET['id'] ?? 0);

$st = $pdo->prepare("SELECT * FROM moradores WHERE id=?");
$st->execute([$id]);
$m = $st->fetch();
if (!$m) { http_response_code(404); exit('Morador não encontrado.'); }

$quartos = $pdo->query("SELECT * FROM quartos ORDER BY andar, codigo")->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $nome = trim((string)($_POST['nome'] ?? ''));
  $doc  = trim((string)($_POST['documento'] ?? ''));
  $novo_quarto_id = (int)($_POST['quarto_id'] ?? 0);

  // ✅ checkbox: segunda pessoa = 2, senão = 1
  $qtd_pessoas = isset($_POST['segunda_pessoa']) ? 2 : 1;

  $banheiro_privativo = isset($_POST['banheiro_privativo']) ? 1 : 0;
  $valor_mensal = calc_valor($banheiro_privativo, $qtd_pessoas);

  if ($nome === '' || $novo_quarto_id <= 0) {
    flash_set('error', 'Informe nome e quarto.');
    redirect('edit.php?id=' . $id);
  }

  $pdo->beginTransaction();
  try {
    $pdo->prepare("
      UPDATE moradores
      SET nome=?, documento=?, quarto_id=?, qtd_pessoas=?, banheiro_privativo=?, valor_mensal=?
      WHERE id=?
    ")->execute([
      $nome,
      $doc !== '' ? $doc : null,
      $novo_quarto_id,
      $qtd_pessoas,
      $banheiro_privativo,
      $valor_mensal,
      $id
    ]);

    // se mudou de quarto, ajusta status do antigo e do novo
    $old_quarto_id = (int)$m['quarto_id'];
    if ($novo_quarto_id !== $old_quarto_id) {
      // marca novo como alugado
      $pdo->prepare("UPDATE quartos SET status='alugado' WHERE id=?")->execute([$novo_quarto_id]);

      // se antigo ficou sem moradores ativos, libera
      $st2 = $pdo->prepare("SELECT COUNT(*) c FROM moradores WHERE quarto_id=? AND ativo=1");
      $st2->execute([$old_quarto_id]);
      if ((int)$st2->fetch()['c'] === 0) {
        $pdo->prepare("UPDATE quartos SET status='disponivel' WHERE id=?")->execute([$old_quarto_id]);
      }
    }

    $pdo->commit();
    flash_set('success','Morador atualizado. Mensalidade: ' . money($valor_mensal));
    redirect('view.php?id='.$id);
  } catch (Throwable $e) {
    $pdo->rollBack();
    flash_set('error','Erro: '.$e->getMessage());
    redirect('edit.php?id='.$id);
  }
}

require_once __DIR__ . '/../../includes/header.php';
require_once __DIR__ . '/../../includes/sidebar.php';

$segunda_pessoa_checked = ((int)($m['qtd_pessoas'] ?? 1) === 2);
?>
<div class="card">
  <div style="font-weight:900;font-size:18px">Editar morador</div>
  <div class="sep"></div>

  <form method="post" class="form-grid cols-2">
    <input type="hidden" name="csrf" value="<?= htmlspecialchars(csrf_token()) ?>">

    <div style="grid-column:1/-1">
      <label>Nome</label>
      <input class="input" name="nome" value="<?= htmlspecialchars((string)$m['nome']) ?>" required>
    </div>

    <div>
      <label>Documento</label>
      <input class="input" name="documento" value="<?= htmlspecialchars((string)($m['documento'] ?? '')) ?>">
    </div>

    <div>
      <label>Quarto</label>
      <select class="input" name="quarto_id" required>
        <?php foreach ($quartos as $q): ?>
          <option value="<?= (int)$q['id'] ?>" <?= (int)$m['quarto_id'] === (int)$q['id'] ? 'selected' : '' ?>>
            <?= htmlspecialchars($q['codigo']) ?> — <?= (int)$q['andar'] ?>º — <?= (int)$q['banheiro_privativo'] ? 'Privativo' : 'Compartilhado' ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>

    <!-- ✅ NOVA CHECKBOX: segunda pessoa (carrega do banco) -->
    <div class="checkbox" style="grid-column:1/-1">
      <input type="checkbox" name="segunda_pessoa" id="sp" <?= $segunda_pessoa_checked ? 'checked' : '' ?>>
      <div>
        <label for="sp" style="margin:0;color:var(--text)">Há uma segunda pessoa morando junto?</label>
        <small>Se marcar, o sistema considera <b>2 pessoas</b> e recalcula a mensalidade automaticamente.</small>
      </div>
    </div>

    <div class="checkbox" style="grid-column:1/-1">
      <input type="checkbox" name="banheiro_privativo" id="bp" <?= (int)($m['banheiro_privativo'] ?? 0) === 1 ? 'checked' : '' ?>>
      <div>
        <label for="bp" style="margin:0;color:var(--text)">Quarto com banheiro privativo (no quarto)</label>
        <small><b>Observação:</b> só marque se o quarto alugado for no <b>primeiro</b> ou <b>segundo</b> andar.</small>
      </div>
    </div>

    <div class="row" style="grid-column:1/-1">
      <button class="btn" type="submit">Salvar</button>
      <a class="btn btn-ghost" href="view.php?id=<?= (int)$m['id'] ?>">Voltar</a>
    </div>
  </form>
</div>
<?php require_once __DIR__ . '/../../includes/footer.php'; ?>