<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/app.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();
csrf_check();

$quartos = $pdo->query("SELECT * FROM quartos ORDER BY andar, codigo")->fetchAll();

function calc_valor(int $banheiro_privativo, int $qtd): float {
  $qtd = ($qtd === 2) ? 2 : 1;
  if ($banheiro_privativo === 1) return PRECO['privativo'][$qtd];
  return PRECO['compartilhado'][$qtd];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $nome = trim((string)($_POST['nome'] ?? ''));
  $doc  = trim((string)($_POST['documento'] ?? ''));
  $quarto_id = (int)($_POST['quarto_id'] ?? 0);

  // ✅ checkbox: segunda pessoa = 2, senão = 1
  $qtd_pessoas = isset($_POST['segunda_pessoa']) ? 2 : 1;

  $banheiro_privativo = isset($_POST['banheiro_privativo']) ? 1 : 0;

  // ✅ telefones
  $telefone_contato = trim((string)($_POST['telefone_contato'] ?? ''));
  $falar_com = trim((string)($_POST['falar_com'] ?? ''));
  $telefone_emergencia = trim((string)($_POST['telefone_emergencia'] ?? ''));

  $pagamento_entrada = (float)str_replace(',', '.', (string)($_POST['pagamento_entrada'] ?? '0'));
  $data_entrada = (string)($_POST['data_entrada'] ?? date('Y-m-d'));
  $forma = trim((string)($_POST['forma'] ?? ''));

  if ($nome === '' || $quarto_id <= 0) {
    flash_set('error', 'Informe nome e quarto.');
    redirect('create.php');
  }

  $valor_mensal = calc_valor($banheiro_privativo, $qtd_pessoas);

  $pdo->beginTransaction();
  try {
    $st = $pdo->prepare("
      INSERT INTO moradores
        (nome, documento, quarto_id, qtd_pessoas, banheiro_privativo, valor_mensal,
         pagamento_entrada, data_entrada, ativo,
         telefone_contato, falar_com, telefone_emergencia)
      VALUES (?,?,?,?,?,?,?,?,1, ?,?,?)
    ");
    $st->execute([
      $nome,
      $doc !== '' ? $doc : null,
      $quarto_id,
      $qtd_pessoas,
      $banheiro_privativo,
      $valor_mensal,
      $pagamento_entrada,
      $data_entrada,
      $telefone_contato !== '' ? $telefone_contato : null,
      $falar_com !== '' ? $falar_com : null,
      $telefone_emergencia !== '' ? $telefone_emergencia : null,
    ]);

    $morador_id = (int)$pdo->lastInsertId();

    // marca quarto como alugado
    $pdo->prepare("UPDATE quartos SET status='alugado' WHERE id=?")->execute([$quarto_id]);

    // cria controle facial padrão
    try {
      $pdo->prepare("INSERT INTO facial_controle (morador_id, cadastrado) VALUES (?,0)")
          ->execute([$morador_id]);
    } catch (Throwable $e) {
      // não trava o cadastro
    }

    // registra pagamento de entrada se informado
    if ($pagamento_entrada > 0) {
      $pdo->prepare("
        INSERT INTO pagamentos (morador_id, tipo, data_pagamento, valor_base, multa, desconto, total, forma, observacao)
        VALUES (?,?,?, ?,0,0, ?,?,?)
      ")->execute([
        $morador_id,
        'entrada',
        $data_entrada,
        $pagamento_entrada,
        $pagamento_entrada,
        $forma !== '' ? $forma : null,
        'Pagamento de entrada'
      ]);
    }

    $pdo->commit();
    flash_set('success', 'Morador cadastrado. Mensalidade: ' . money($valor_mensal));
    redirect('index.php');
  } catch (Throwable $e) {
    $pdo->rollBack();
    flash_set('error', 'Erro: ' . $e->getMessage());
    redirect('create.php');
  }
}

require_once __DIR__ . '/../../includes/header.php';
require_once __DIR__ . '/../../includes/sidebar.php';
?>
<div class="card">
  <div style="font-weight:900;font-size:18px">Novo morador</div>
  <div style="color:var(--muted);font-size:13px;margin-top:4px">Valores padrão aplicados automaticamente</div>
  <div class="sep"></div>

  <div class="card" style="background:rgba(255,255,255,.01)">
    <div style="font-weight:800">Valores padrão</div>
    <div style="color:var(--muted);font-size:13px;margin-top:6px">
      <ul style="margin:0;padding-left:18px">
        <li>Banheiro compartilhado: 1 pessoa <?= money(600) ?> • 2 pessoas <?= money(800) ?></li>
        <li>Banheiro privativo: 1 pessoa <?= money(1000) ?> • 2 pessoas <?= money(1200) ?></li>
      </ul>
    </div>
  </div>

  <div class="sep"></div>

  <form method="post" class="form-grid cols-2">
    <input type="hidden" name="csrf" value="<?= htmlspecialchars(csrf_token()) ?>">

    <div style="grid-column:1/-1">
      <label>Nome do morador</label>
      <input class="input" name="nome" required>
    </div>

    <div>
      <label>Documento (opcional)</label>
      <input class="input" name="documento" placeholder="CPF/RG (opcional)">
    </div>

    <div>
      <label>Quarto</label>
      <select class="input" name="quarto_id" required>
        <option value="">Selecione…</option>
        <?php foreach ($quartos as $q): ?>
          <option value="<?= (int)$q['id'] ?>">
            <?= htmlspecialchars($q['codigo']) ?> — <?= (int)$q['andar'] ?>º — <?= (int)$q['banheiro_privativo'] ? 'Privativo' : 'Compartilhado' ?>
            — <?= $q['status']==='alugado' ? '(Alugado)' : '(Disponível)' ?>
          </option>
        <?php endforeach; ?>
      </select>
      <div style="color:var(--muted);font-size:12px;margin-top:6px">
        Dica: você pode cadastrar mesmo se estiver “Alugado” (caso tenha 2 pessoas no mesmo quarto).
      </div>
    </div>

    <!-- ✅ CHECKBOX: segunda pessoa -->
    <div class="checkbox" style="grid-column:1/-1">
      <input type="checkbox" name="segunda_pessoa" id="sp">
      <div>
        <label for="sp" style="margin:0;color:var(--text)">Há uma segunda pessoa morando junto?</label>
        <small>Se marcar, o sistema considera <b>2 pessoas</b> e recalcula a mensalidade automaticamente.</small>
      </div>
    </div>

    <div class="checkbox" style="grid-column:1/-1">
      <input type="checkbox" name="banheiro_privativo" id="bp">
      <div>
        <label for="bp" style="margin:0;color:var(--text)">Quarto com banheiro privativo (no quarto)</label>
        <small><b>Observação:</b> só marque se o quarto alugado for no <b>primeiro</b> ou <b>segundo</b> andar.</small>
      </div>
    </div>

    <!-- ✅ Contatos -->
    <div>
      <label>Telefone de contato</label>
      <input class="input" name="telefone_contato" placeholder="Ex: (11) 91234-5678">
    </div>

    <div>
      <label>Falar com</label>
      <input class="input" name="falar_com" placeholder="Ex: João / Mãe / Responsável">
    </div>

    <div>
      <label>Telefone de emergência</label>
      <input class="input" name="telefone_emergencia" placeholder="Ex: (11) 90000-0000">
    </div>

    <div>
      <label>Pagamento da entrada (R$)</label>
      <input class="input" name="pagamento_entrada" placeholder="Ex: 200,00">
    </div>

    <div>
      <label>Forma de pagamento (entrada)</label>
      <select class="input" name="forma">
        <option value="">—</option>
        <option value="pix">PIX</option>
        <option value="dinheiro">Dinheiro</option>
        <option value="cartao">Cartão</option>
        <option value="transferencia">Transferência</option>
      </select>
    </div>

    <div>
      <label>Data de entrada</label>
      <input class="input" type="date" name="data_entrada" value="<?= date('Y-m-d') ?>" required>
    </div>

    <div class="row" style="grid-column:1/-1">
      <button class="btn" type="submit">Salvar</button>
      <a class="btn btn-ghost" href="index.php">Cancelar</a>
    </div>
  </form>
</div>
<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
