<?php
declare(strict_types=1);

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/app.php';
require_once __DIR__ . '/includes/functions.php';

require_login();

// Cards básicos
$total_quartos   = (int)$pdo->query("SELECT COUNT(*) FROM quartos")->fetchColumn();
$moradores_ativos = (int)$pdo->query("SELECT COUNT(*) FROM moradores WHERE ativo=1")->fetchColumn();
$quartos_alugados = (int)$pdo->query("SELECT COUNT(*) FROM quartos WHERE status='alugado'")->fetchColumn();
$quartos_disp     = (int)$pdo->query("SELECT COUNT(*) FROM quartos WHERE status='disponivel'")->fetchColumn();
$visitas_hoje     = (int)$pdo->query("SELECT COUNT(*) FROM visitas WHERE DATE(data_entrada)=CURDATE()")->fetchColumn();

/**
 * ✅ RECEBIMENTOS robustos
 */
function pagamentos_has_col(PDO $pdo, string $col): bool {
  $st = $pdo->prepare("
    SELECT COUNT(*)
    FROM information_schema.columns
    WHERE table_schema = DATABASE()
      AND table_name = 'pagamentos'
      AND column_name = ?
  ");
  $st->execute([$col]);
  return (int)$st->fetchColumn() > 0;
}

$has_total      = pagamentos_has_col($pdo, 'total');
$has_valorTotal = pagamentos_has_col($pdo, 'valor_total');
$has_valorBase  = pagamentos_has_col($pdo, 'valor_base');
$has_valor      = pagamentos_has_col($pdo, 'valor');
$has_multa      = pagamentos_has_col($pdo, 'multa');
$has_desconto   = pagamentos_has_col($pdo, 'desconto');

if ($has_total) {
  $expr_total_real = 'COALESCE(p.total,0)';
} elseif ($has_valorTotal) {
  $expr_total_real = 'COALESCE(p.valor_total,0)';
} else {
  $principal = $has_valorBase ? 'COALESCE(p.valor_base,0)' : ($has_valor ? 'COALESCE(p.valor,0)' : '0');
  $multa     = $has_multa ? 'COALESCE(p.multa,0)' : '0';
  $desconto  = $has_desconto ? 'COALESCE(p.desconto,0)' : '0';
  $expr_total_real = "($principal + $multa - $desconto)";
}

$recebido_mes = (float)$pdo->query("
  SELECT COALESCE(SUM($expr_total_real),0)
  FROM pagamentos p
  WHERE p.data_pagamento IS NOT NULL
    AND YEAR(DATE(p.data_pagamento)) = YEAR(CURDATE())
    AND MONTH(DATE(p.data_pagamento)) = MONTH(CURDATE())
")->fetchColumn();

$recebido_ano = (float)$pdo->query("
  SELECT COALESCE(SUM($expr_total_real),0)
  FROM pagamentos p
  WHERE p.data_pagamento IS NOT NULL
    AND YEAR(DATE(p.data_pagamento)) = YEAR(CURDATE())
")->fetchColumn();

$multas_mes = $has_multa ? (float)$pdo->query("
  SELECT COALESCE(SUM(COALESCE(multa,0)),0)
  FROM pagamentos
  WHERE data_pagamento IS NOT NULL
    AND YEAR(DATE(data_pagamento))=YEAR(CURDATE())
    AND MONTH(DATE(data_pagamento))=MONTH(CURDATE())
")->fetchColumn() : 0.0;

$descontos_mes = $has_desconto ? (float)$pdo->query("
  SELECT COALESCE(SUM(COALESCE(desconto,0)),0)
  FROM pagamentos
  WHERE data_pagamento IS NOT NULL
    AND YEAR(DATE(data_pagamento))=YEAR(CURDATE())
    AND MONTH(DATE(data_pagamento))=MONTH(CURDATE())
")->fetchColumn() : 0.0;

// Visitas longas (>=5 dias)
$visitas_longas = $pdo->query("
  SELECT v.id, v.nome_visitante, v.telefone, v.data_entrada,
         m.nome AS morador_nome, q.codigo AS quarto,
         DATEDIFF(NOW(), v.data_entrada) AS dias
  FROM visitas v
  JOIN moradores m ON m.id = v.morador_id
  JOIN quartos q ON q.id = m.quarto_id
  WHERE DATEDIFF(NOW(), v.data_entrada) >= 5
  ORDER BY dias DESC, v.data_entrada ASC
  LIMIT 50
")->fetchAll();

// Visitas de hoje
$visitas_de_hoje = $pdo->query("
  SELECT v.id, v.nome_visitante, v.telefone, v.data_entrada,
         m.nome AS morador_nome, q.codigo AS quarto
  FROM visitas v
  JOIN moradores m ON m.id = v.morador_id
  JOIN quartos q ON q.id = m.quarto_id
  WHERE DATE(v.data_entrada)=CURDATE()
  ORDER BY v.data_entrada DESC, v.id DESC
  LIMIT 100
")->fetchAll();

require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/sidebar.php';

// ✅ Grid responsivo (não mexe no CSS do projeto)
$gridStyle = 'display:grid;grid-template-columns:repeat(auto-fit,minmax(220px,1fr));gap:14px;align-items:stretch;';
?>

<div class="row" style="justify-content:space-between;align-items:center">
  <div>
    <div style="font-weight:900;font-size:18px">Dashboard</div>
    <div style="color:var(--muted);font-size:13px;margin-top:4px">
      Visão geral do pensionato + visitas + recebimentos
    </div>
  </div>
  <div class="row">
    <a class="btn" href="/pensionato/modules/visitas/create.php">+ Registrar visita</a>
    <a class="btn btn-ghost" href="/pensionato/modules/pagamentos/create.php">+ Pagamento</a>
  </div>
</div>

<div class="sep"></div>

<!-- ✅ Cards principais -->
<div style="<?= $gridStyle ?>">
  <div class="card">
    <div style="color:var(--muted);font-size:12px">Total de quartos</div>
    <div style="font-weight:900;font-size:22px;margin-top:6px"><?= (int)$total_quartos ?></div>
  </div>

  <div class="card">
    <div style="color:var(--muted);font-size:12px">Moradores ativos</div>
    <div style="font-weight:900;font-size:22px;margin-top:6px"><?= (int)$moradores_ativos ?></div>
  </div>

  <div class="card">
    <div style="color:var(--muted);font-size:12px">Quartos alugados</div>
    <div style="font-weight:900;font-size:22px;margin-top:6px"><?= (int)$quartos_alugados ?></div>
  </div>

  <div class="card">
    <div style="color:var(--muted);font-size:12px">Quartos disponíveis</div>
    <div style="font-weight:900;font-size:22px;margin-top:6px"><?= (int)$quartos_disp ?></div>
  </div>

  <div class="card">
    <div style="color:var(--muted);font-size:12px">Visitas hoje</div>
    <div style="font-weight:900;font-size:22px;margin-top:6px"><?= (int)$visitas_hoje ?></div>
  </div>
</div>

<div class="sep"></div>

<!-- ✅ Recebimentos -->
<div style="<?= $gridStyle ?>">
  <div class="card">
    <div style="color:var(--muted);font-size:12px">Recebido no mês (<?= date('m/Y') ?>)</div>
    <div style="font-weight:900;font-size:26px;margin-top:6px"><?= money($recebido_mes) ?></div>
    <div style="color:var(--muted);font-size:12px;margin-top:6px">
      Mensalidade/Entrada/Outros + Multa − Desconto
    </div>
  </div>

  <div class="card">
    <div style="color:var(--muted);font-size:12px">Recebido no ano (<?= date('Y') ?>)</div>
    <div style="font-weight:900;font-size:26px;margin-top:6px"><?= money($recebido_ano) ?></div>
    <div style="color:var(--muted);font-size:12px;margin-top:6px">
      Total real no ano
    </div>
  </div>

  <div class="card">
    <div style="color:var(--muted);font-size:12px">Multas no mês</div>
    <div style="font-weight:900;font-size:22px;margin-top:6px"><?= money($multas_mes) ?></div>
  </div>

  <div class="card">
    <div style="color:var(--muted);font-size:12px">Descontos no mês</div>
    <div style="font-weight:900;font-size:22px;margin-top:6px"><?= money($descontos_mes) ?></div>
  </div>
</div>

<div class="sep"></div>

<!-- ✅ Alertas + Visitas de hoje -->
<div style="display:grid;grid-template-columns:1.2fr .8fr;gap:14px;align-items:start;">
  <div class="card">
    <div class="row" style="justify-content:space-between">
      <div style="font-weight:900">Alertas de visitas (5+ dias)</div>
      <div class="row">
        <a class="btn btn-ghost" href="/pensionato/modules/visitas/index.php?todas=1">Ver todas</a>
        <a class="btn btn-ghost" href="/pensionato/modules/visitas/index.php">Ver hoje</a>
      </div>
    </div>

    <div class="sep"></div>

    <?php if (!$visitas_longas): ?>
      <div style="color:var(--muted)">Nenhuma visita com mais de 5 dias.</div>
    <?php else: ?>
      <table class="table">
        <thead>
          <tr>
            <th>Visitante</th>
            <th>Morador</th>
            <th>Quarto</th>
            <th>Entrada</th>
            <th>Dias</th>
            <th>Adicional</th>
            <th>Ação</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($visitas_longas as $v): ?>
            <?php
              $dias = (int)$v['dias'];
              $extra_dias = max(0, $dias - 5);
              $adicional = $extra_dias * 200;

              $badge = $dias >= 7
                ? '<span class="badge danger">7+ dias</span>'
                : '<span class="badge warn">5-6 dias</span>';
            ?>
            <tr>
              <td><?= htmlspecialchars($v['nome_visitante']) ?> <?= $badge ?></td>
              <td><?= htmlspecialchars($v['morador_nome']) ?></td>
              <td><b><?= htmlspecialchars($v['quarto']) ?></b></td>
              <td><?= htmlspecialchars($v['data_entrada']) ?></td>
              <td><b><?= $dias ?></b></td>
              <td><b><?= money((float)$adicional) ?></b></td>
              <td>
                <form method="post" action="/pensionato/modules/visitas/delete.php"
                      style="display:inline"
                      onsubmit="return confirm('Remover visita? (visita foi embora)');">
                  <input type="hidden" name="csrf" value="<?= htmlspecialchars(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$v['id'] ?>">
                  <button class="btn btn-ghost" type="submit">Remover</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>

      <div style="margin-top:10px;color:var(--muted);font-size:12px">
        Regra: após <b>5 dias</b>, adicional de <b>R$ 200/dia</b> (do 6º dia em diante).
      </div>
    <?php endif; ?>
  </div>

  <div class="card">
    <div class="row" style="justify-content:space-between">
      <div style="font-weight:900">Visitas de hoje</div>
      <a class="btn btn-ghost" href="/pensionato/modules/visitas/index.php">Abrir módulo</a>
    </div>

    <div class="sep"></div>

    <?php if (!$visitas_de_hoje): ?>
      <div style="color:var(--muted)">Nenhuma visita registrada hoje.</div>
    <?php else: ?>
      <table class="table">
        <thead>
          <tr>
            <th>Entrada</th>
            <th>Visitante</th>
            <th>Morador</th>
            <th>Quarto</th>
            <th></th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($visitas_de_hoje as $v): ?>
            <tr>
              <td><?= htmlspecialchars($v['data_entrada']) ?></td>
              <td><?= htmlspecialchars($v['nome_visitante']) ?></td>
              <td><?= htmlspecialchars($v['morador_nome']) ?></td>
              <td><b><?= htmlspecialchars($v['quarto']) ?></b></td>
              <td>
                <form method="post" action="/pensionato/modules/visitas/delete.php"
                      style="display:inline"
                      onsubmit="return confirm('Remover visita? (visita foi embora)');">
                  <input type="hidden" name="csrf" value="<?= htmlspecialchars(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$v['id'] ?>">
                  <button class="btn btn-ghost" type="submit">Remover</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    <?php endif; ?>
  </div>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
