<?php
declare(strict_types=1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/app.php';
require_once __DIR__ . '/../includes/functions.php';

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

// Se já estiver logado
if (!empty($_SESSION['user_id'])) {
  redirect('/pensionato/dashboard.php');
}

$erro = null;

function normalize_cargo(string $raw): string {
  $v = mb_strtolower(trim($raw), 'UTF-8');
  $v = preg_replace('/[•\|\-\_\/]+/u', ' ', $v);
  $v = preg_replace('/\s+/u', ' ', $v);

  if (strpos($v, 'admin') !== false || strpos($v, 'administrador') !== false || preg_match('/\badm\b/u', $v)) {
    return 'admin';
  }
  if (strpos($v, 'recep') !== false) {
    return 'recepcao';
  }
  return 'recepcao';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $usuario = trim((string)($_POST['usuario'] ?? ''));
  $senha   = (string)($_POST['senha'] ?? '');

  if ($usuario === '' || $senha === '') {
    $erro = 'Informe usuário e senha.';
  } else {
    $st = $pdo->prepare("
      SELECT id, nome, usuario, email, senha_hash, cargo, ativo, must_reset_password
      FROM usuarios
      WHERE usuario = ?
      LIMIT 1
    ");
    $st->execute([$usuario]);
    $u = $st->fetch();

    if (!$u || (int)$u['ativo'] !== 1 || !password_verify($senha, (string)$u['senha_hash'])) {
      $erro = 'Usuário ou senha inválidos.';
    } else {
      $cargo = normalize_cargo((string)($u['cargo'] ?? ''));

      $_SESSION['user_id'] = (int)$u['id'];
      $_SESSION['user'] = [
        'id' => (int)$u['id'],
        'nome' => (string)$u['nome'],
        'usuario' => (string)$u['usuario'],
        'email' => (string)($u['email'] ?? ''),
        'cargo' => $cargo,
      ];
      $_SESSION['cargo'] = $cargo;

      if ((int)$u['must_reset_password'] === 1) {
        redirect('/pensionato/modules/usuarios/force_reset.php');
      }

      redirect('/pensionato/dashboard.php');
    }
  }
}
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
  <meta charset="UTF-8">
  <title>Login • <?= htmlspecialchars(APP_NAME) ?></title>

  <!-- 🔥 FORÇA CENTRALIZAÇÃO TOTAL -->
  <style>
    html, body {
      width: 100%;
      height: 100%;
      margin: 0;
    }
    body {
      display: flex;
      align-items: center;
      justify-content: center;
      background: var(--bg, #0f1115);
      font-family: system-ui, -apple-system, BlinkMacSystemFont, sans-serif;
    }
    .login-card {
      width: 420px;
      max-width: 92vw;
      background: #151821;
      border-radius: 10px;
      padding: 28px;
      box-shadow: 0 20px 60px rgba(0,0,0,.45);
    }
  </style>

  <!-- mantém seus estilos globais -->
  <link rel="stylesheet" href="/pensionato/assets/css/app.css">
</head>
<body>

  <div class="login-card">

    <!-- LOGO -->
    <div style="display:flex;justify-content:center;margin-bottom:16px">
      <img src="/pensionato/assets/imagens/logo.png.png" alt="Logo" style="max-height:80px">
    </div>

    <div style="font-weight:900;font-size:20px;text-align:center">
      <?= htmlspecialchars(APP_NAME) ?>
    </div>

    <div style="color:var(--muted);font-size:13px;text-align:center;margin-top:6px">
      Acesso ao sistema
    </div>

    <div class="sep"></div>

    <?php if ($erro): ?>
      <div class="card" style="background:rgba(255,0,0,.08);border:1px solid rgba(255,0,0,.25)">
        <?= htmlspecialchars($erro) ?>
      </div>
      <div class="sep"></div>
    <?php endif; ?>

    <form method="post" class="form-grid cols-1">
      <div>
        <label>Usuário</label>
        <input class="input" name="usuario" required autofocus>
      </div>

      <div>
        <label>Senha</label>
        <input class="input" type="password" name="senha" required>
      </div>

      <div class="row" style="margin-top:16px">
        <button class="btn" type="submit" style="flex:1">Entrar</button>
      </div>
    </form>

    <div style="margin-top:16px;text-align:center;color:var(--muted);font-size:12px">
      Primeiro acesso? Você será solicitado a redefinir a senha.
    </div>

  </div>

</body>
</html>
