/**
 * Melhorias de Responsividade Mobile
 * Adiciona funcionalidades para melhor UX em dispositivos móveis
 */

document.addEventListener('DOMContentLoaded', function() {
  // ===== Detecção de Dispositivo =====
  const isMobile = () => window.innerWidth <= 768;
  const isSmallMobile = () => window.innerWidth <= 480;

  // ===== Prevent Zoom na Dupla Tap =====
  let lastTouchEnd = 0;
  document.addEventListener('touchend', (e) => {
    const now = Date.now();
    if (now - lastTouchEnd <= 300) {
      e.preventDefault();
    }
    lastTouchEnd = now;
  }, false);

  // ===== Input Focus Management =====
  const inputs = document.querySelectorAll('input, select, textarea');
  inputs.forEach(input => {
    input.addEventListener('focus', () => {
      // Scroll para o input em mobile
      if (isMobile()) {
        setTimeout(() => {
          input.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }, 300);
      }
    });
  });

  // ===== Table Horizontal Scroll Indicator =====
  const tables = document.querySelectorAll('.table-wrapper');
  tables.forEach(wrapper => {
    wrapper.addEventListener('scroll', () => {
      if (wrapper.scrollLeft > 0) {
        wrapper.classList.add('scrolled-left');
      } else {
        wrapper.classList.remove('scrolled-left');
      }
      
      if (wrapper.scrollLeft + wrapper.clientWidth < wrapper.scrollWidth) {
        wrapper.classList.add('can-scroll-right');
      } else {
        wrapper.classList.remove('can-scroll-right');
      }
    });

    // Trigger initial check
    wrapper.dispatchEvent(new Event('scroll'));
  });

  // ===== Modal Responsivo =====
  const createResponsiveModal = () => {
    const modals = document.querySelectorAll('[role="dialog"]');
    modals.forEach(modal => {
      if (isMobile()) {
        modal.style.maxWidth = '95vw';
        modal.style.maxHeight = '95vh';
      }
    });
  };

  // ===== Keyboard Detection =====
  let keyboardVisible = false;
  window.addEventListener('resize', () => {
    const currentHeight = window.innerHeight;
    if (currentHeight < 500) {
      keyboardVisible = true;
    } else {
      keyboardVisible = false;
    }
  });

  // ===== Form Submission Feedback =====
  const forms = document.querySelectorAll('form');
  forms.forEach(form => {
    form.addEventListener('submit', () => {
      // Desabilita botões de submit para evitar dupla submissão
      const buttons = form.querySelectorAll('button[type="submit"]');
      buttons.forEach(btn => {
        btn.disabled = true;
        btn.style.opacity = '0.6';
        btn.textContent = 'Carregando...';
      });
    });
  });

  // ===== Detecta Viewport Changes =====
  let lastVW = window.innerWidth;
  window.addEventListener('orientationchange', () => {
    createResponsiveModal();
  });

  window.addEventListener('resize', () => {
    const currentVW = window.innerWidth;
    if (Math.abs(currentVW - lastVW) > 100) {
      // Viewport mudou significativamente
      lastVW = currentVW;
      createResponsiveModal();
    }
  });

  // ===== Touch Feedback =====
  document.addEventListener('touchstart', function(e) {
    if (e.target.classList.contains('btn') || e.target.classList.contains('menu-item')) {
      e.target.style.opacity = '0.7';
    }
  }, true);

  document.addEventListener('touchend', function(e) {
    if (e.target.classList.contains('btn') || e.target.classList.contains('menu-item')) {
      e.target.style.opacity = '1';
    }
  }, true);

  // ===== Prevent Text Selection on Double Click =====
  document.addEventListener('touchstart', function(e) {
    if (e.touches.length > 1) {
      e.preventDefault();
    }
  }, { passive: false });

  // ===== Safe Area Padding para Notch Devices =====
  if (CSS.supports('padding-left: max(0px)')) {
    const shell = document.getElementById('app-shell');
    if (shell) {
      shell.style.paddingLeft = 'max(0px, env(safe-area-inset-left))';
      shell.style.paddingRight = 'max(0px, env(safe-area-inset-right))';
      shell.style.paddingTop = 'max(0px, env(safe-area-inset-top))';
      shell.style.paddingBottom = 'max(0px, env(safe-area-inset-bottom))';
    }
  }

  console.log('Responsive improvements loaded ✓');
});
